//#-hidden-code
//
//  See LICENSE folder for this template’s licensing information.
//
//  Abstract:
//  The Swift file containing the source code edited by the user of this playground book.
//

//#-end-hidden-code
/*: some text
 ##  ホログラム
 立体映像を作ってみよう\
 🔸立体視するためには、ピラミッド型のアクリル反射板を用います。
 * Important:
 usdz形式ファイルは立体オブジェクトファイルです。\
 必要に応じて、各種パラメーターを調整してください。
 ---
 */
//#-code-completion(everything, hide)
//#-editable-code
import UIKit
import PlaygroundSupport
import SceneKit

// シーンの設定
let scene = try SCNScene(url: #fileLiteral(resourceName: "MUNEATSU3D-7.usdz"))

//画角スケール調整(大きほど小さく見える）
let graphicScale = 0.25

// ビューの初期サイズ
let viewSize = CGSize(width: 300, height: 300)
let containerViewSize = CGSize(width: 600, height: 600)

// 対角線を描画するビュー
public class DiagonalView: UIView {
    override public func draw(_ rect: CGRect) {
        super.draw(rect)
        guard let context = UIGraphicsGetCurrentContext() else { return }
        
        // 対角線の色と幅を設定
        context.setStrokeColor(UIColor.red.cgColor)
        context.setLineWidth(2.0)
        
        // 左上から右下への対角線
        context.move(to: CGPoint(x: 0, y: 0))
        context.addLine(to: CGPoint(x: rect.width, y: rect.height))
        context.strokePath()
        
        // 右上から左下への対角線
        context.move(to: CGPoint(x: rect.width, y: 0))
        context.addLine(to: CGPoint(x: 0, y: rect.height))
        context.strokePath()
    }
}


public class myLiveView: UIView,PlaygroundLiveViewSafeAreaContainer {}
let liveView = myLiveView()
liveView.backgroundColor = UIColor.black
let containerView = DiagonalView()
containerView.clipsToBounds = true
containerView.translatesAutoresizingMaskIntoConstraints = false
liveView.addSubview(containerView)
NSLayoutConstraint.activate([
    containerView.centerXAnchor.constraint(equalTo: liveView.liveViewSafeAreaGuide.centerXAnchor),
    containerView.centerYAnchor.constraint(equalTo: liveView.liveViewSafeAreaGuide.centerYAnchor),
    containerView.widthAnchor.constraint(equalTo: liveView.liveViewSafeAreaGuide.widthAnchor),
    containerView.heightAnchor.constraint(equalTo: containerView.widthAnchor)
    ])

// アニメーションの再生を開始する関数
func playAnimation(for node: SCNNode) {
    // すべてのアニメーションキーを取得
    let animationKeys = node.animationKeys
    
    for key in animationKeys {
        if let animation = node.animation(forKey: key) {
            node.addAnimation(animation, forKey: key)
        }
    }
}

// アニメーションを再生する
if let modelNode = scene.rootNode.childNodes.first {
    playAnimation(for: modelNode)
}

// オブジェクトのバウンディングボックスを取得
let (min, max) = scene.rootNode.boundingBox
let objectHeight = max.y - min.y

// ビュー1の設定（前方）
let sceneViewFront = SCNView(frame: CGRect(x: (containerViewSize.width - viewSize.width) / 2, y: containerViewSize.height - viewSize.height, width: viewSize.width, height: viewSize.height))
sceneViewFront.autoenablesDefaultLighting = true
sceneViewFront.backgroundColor = .clear
sceneViewFront.scene = scene
sceneViewFront.clipsToBounds = true
sceneViewFront.translatesAutoresizingMaskIntoConstraints = false

// ビュー2の設定（後方）
let sceneViewBack = SCNView(frame: CGRect(x: (containerViewSize.width - viewSize.width) / 2, y: 0, width: viewSize.width, height: viewSize.height))
sceneViewBack.autoenablesDefaultLighting = true
sceneViewBack.backgroundColor = .clear
sceneViewBack.scene = scene
sceneViewBack.clipsToBounds = true
sceneViewBack.translatesAutoresizingMaskIntoConstraints = false

// ビュー3の設定（右側）
let sceneViewRight = SCNView(frame: CGRect(x: containerViewSize.width - viewSize.width, y: (containerViewSize.height - viewSize.height) / 2, width: viewSize.width, height: viewSize.height))
sceneViewRight.autoenablesDefaultLighting = true
sceneViewRight.backgroundColor = .clear
sceneViewRight.scene = scene
// SCNViewを90度回転
sceneViewRight.transform = CGAffineTransform(rotationAngle: CGFloat(Double.pi / 2))
sceneViewRight.clipsToBounds = true
sceneViewRight.translatesAutoresizingMaskIntoConstraints = false

// ビュー4の設定（左側）
let sceneViewLeft = SCNView(frame: CGRect(x: 0, y: (containerViewSize.height - viewSize.height) / 2, width: viewSize.width, height: viewSize.height))
sceneViewLeft.autoenablesDefaultLighting = true
sceneViewLeft.backgroundColor = .clear
sceneViewLeft.scene = scene
// SCNViewを90度回転
sceneViewLeft.transform = CGAffineTransform(rotationAngle: CGFloat(-Double.pi / 2))
sceneViewLeft.clipsToBounds = true
sceneViewLeft.translatesAutoresizingMaskIntoConstraints = false

// カメラの設定関数
func setupCamera(for sceneView: SCNView, position: SCNVector3, eulerAngles: SCNVector3) {
    let cameraNode = SCNNode()
    let camera = SCNCamera()
    camera.usesOrthographicProjection = true // 平行透視を使用
    camera.orthographicScale = Double(graphicScale) // スケールを設定
    camera.zNear = 1.0 // 近接クリッピングプレーン
    camera.zFar = 1000.0 // 遠方クリッピングプレーン
    cameraNode.camera = camera
    cameraNode.position = position
    cameraNode.eulerAngles = eulerAngles
    
    // シーンにカメラを追加
    sceneView.pointOfView = cameraNode
}

// ライトの追加
func addLights(to scene: SCNScene) {
    let ambientLightNode = SCNNode()
    let ambientLight = SCNLight()
    ambientLight.type = .ambient
    ambientLight.intensity = 500
    ambientLightNode.light = ambientLight
    scene.rootNode.addChildNode(ambientLightNode)
    
    let directionalLightNode = SCNNode()
    let directionalLight = SCNLight()
    directionalLight.type = .directional
    directionalLight.intensity = 1000
    directionalLightNode.light = directionalLight
    directionalLightNode.eulerAngles = SCNVector3(-Float.pi / 3, Float.pi / 4, 0)
    scene.rootNode.addChildNode(directionalLightNode)
}

// シーンにライトを追加
addLights(to: scene)

// 各ビューにカメラを設定
setupCamera(for: sceneViewFront, position: SCNVector3(x: 0, y: Float(objectHeight)/2.0, z: 100), eulerAngles: SCNVector3(0, 0, Float.pi))
setupCamera(for: sceneViewBack, position: SCNVector3(x: 0, y: Float(objectHeight)/2.0, z: -100), eulerAngles: SCNVector3(0, Float.pi, 0))
setupCamera(for: sceneViewRight, position: SCNVector3(x: -100, y: Float(objectHeight)/2.0, z: 0), eulerAngles: SCNVector3(0, -Float.pi / 2, 0))
setupCamera(for: sceneViewLeft, position: SCNVector3(x: 100, y: Float(objectHeight)/2.0, z: 0), eulerAngles: SCNVector3(0, Float.pi / 2, 0))

containerView.addSubview(sceneViewFront)
containerView.addSubview(sceneViewBack)
containerView.addSubview(sceneViewRight)
containerView.addSubview(sceneViewLeft)

// 各ビューの制約を追加
NSLayoutConstraint.activate([
    sceneViewFront.centerXAnchor.constraint(equalTo: containerView.centerXAnchor),
    sceneViewFront.bottomAnchor.constraint(equalTo: containerView.bottomAnchor),
    sceneViewFront.widthAnchor.constraint(equalToConstant: viewSize.width),
    sceneViewFront.heightAnchor.constraint(equalToConstant: viewSize.height),
    
    sceneViewBack.centerXAnchor.constraint(equalTo: containerView.centerXAnchor),
    sceneViewBack.topAnchor.constraint(equalTo: containerView.topAnchor),
    sceneViewBack.widthAnchor.constraint(equalToConstant: viewSize.width),
    sceneViewBack.heightAnchor.constraint(equalToConstant: viewSize.height),
    
    sceneViewRight.centerYAnchor.constraint(equalTo: containerView.centerYAnchor),
    sceneViewRight.trailingAnchor.constraint(equalTo: containerView.trailingAnchor),
    sceneViewRight.widthAnchor.constraint(equalToConstant: viewSize.width),
    sceneViewRight.heightAnchor.constraint(equalToConstant: viewSize.height),
    
    sceneViewLeft.centerYAnchor.constraint(equalTo: containerView.centerYAnchor),
    sceneViewLeft.leadingAnchor.constraint(equalTo: containerView.leadingAnchor),
    sceneViewLeft.widthAnchor.constraint(equalToConstant: viewSize.width),
    sceneViewLeft.heightAnchor.constraint(equalToConstant: viewSize.height)
])

PlaygroundPage.current.liveView = liveView
PlaygroundPage.current.needsIndefiniteExecution = true

//#-end-editable-code
//#-hidden-code
//#-end-hidden-code
