//#-hidden-code
//
//  See LICENSE folder for this template’s licensing information.
//
//  Abstract:
//  The Swift file containing the source code edited by the user of this playground book.
//

//#-end-hidden-code
/*: some text
 ##  ホログラム
 立体映像を作ってみよう(ARView)\
 🔸立体視するためには、ピラミッド型のアクリル反射板を用います。
 * Important:
 usdz形式ファイルは立体オブジェクトファイルです。\
 必要に応じて、各種パラメーターを調整してください。
 ---
 */
//#-code-completion(everything, hide)
//#-editable-code
import PlaygroundSupport
import UIKit
import RealityKit

// ARViewの表示エリアを正方形にするためのカスタムビューコントローラー
class ARViewController: UIViewController {
    var arview: ARView!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // ARViewの初期化
        arview = ARView(frame: .zero, cameraMode: .nonAR, automaticallyConfigureSession: true)
        arview.environment.lighting.intensityExponent = 3
        arview.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(arview)
        
        // ARViewの制約を設定（親ビューに対して中心に配置し、正方形になるように）
        let size = min(view.frame.width, view.frame.height)
        NSLayoutConstraint.activate([
            arview.widthAnchor.constraint(equalToConstant: size),
            arview.heightAnchor.constraint(equalTo: arview.widthAnchor),
            arview.centerXAnchor.constraint(equalTo: view.centerXAnchor),
            arview.centerYAnchor.constraint(equalTo: view.centerYAnchor)
        ])
        
        let newAnchor = AnchorEntity(world: .zero)
        let fileUrl = #fileLiteral(resourceName: "MUNEATSU3D-7.usdz")

        let frontEnt = try! Entity.load(contentsOf: fileUrl)
        let leftEnt = try! Entity.load(contentsOf: fileUrl)
        let rightEnt = try! Entity.load(contentsOf: fileUrl)
        let backEnt = try! Entity.load(contentsOf: fileUrl)

        // スケールの設定
        let scale: SIMD3<Float> = SIMD3(x: 1, y: 1, z: 1)
        // スケールを適用
        frontEnt.scale = scale
        leftEnt.scale = scale
        rightEnt.scale = scale
        backEnt.scale = scale

        // 正面
        frontEnt.orientation = simd_quatf(angle: .pi, axis: SIMD3(x: 0, y: 0, z: 1))
        frontEnt.position = SIMD3(x: 0, y: -0.2, z: 0)

        // 左向き
        var xRotation = simd_quatf(angle: .pi / 2, axis: SIMD3(x: 1, y: 0, z: 0))
        var yRotation = simd_quatf(angle: .pi / 2, axis: SIMD3(x: 0, y: 1, z: 0))
        leftEnt.orientation = simd_mul(yRotation, xRotation)
        leftEnt.position = SIMD3(x: 0.2, y: 0, z: 0)

        // 右向き
        xRotation = simd_quatf(angle: .pi / 2, axis: SIMD3(x: 1, y: 0, z: 0))
        yRotation = simd_quatf(angle: -.pi / 2, axis: SIMD3(x: 0, y: 1, z: 0))
        rightEnt.orientation = simd_mul(yRotation, xRotation)
        rightEnt.position = SIMD3(x: -0.2, y: 0, z: 0)

        // 後ろ向き
        backEnt.position = SIMD3(x: -0.2, y: 0, z: 0)
        backEnt.orientation = simd_quatf(angle: .pi, axis: SIMD3(x: 0, y: 1, z: 0))
        backEnt.position = SIMD3(x: 0, y: 0.2, z: 0)

        newAnchor.addChild(frontEnt)
        newAnchor.addChild(leftEnt)
        newAnchor.addChild(rightEnt)
        newAnchor.addChild(backEnt)

        arview.scene.addAnchor(newAnchor)
        for anim in frontEnt.availableAnimations {
            frontEnt.playAnimation(anim.repeat(duration: .infinity), transitionDuration: 1.25, startsPaused: false)
        }
        for anim in leftEnt.availableAnimations {
            leftEnt.playAnimation(anim.repeat(duration: .infinity), transitionDuration: 1.25, startsPaused: false)
        }
        for anim in rightEnt.availableAnimations {
            rightEnt.playAnimation(anim.repeat(duration: .infinity), transitionDuration: 1.25, startsPaused: false)
        }
        for anim in backEnt.availableAnimations {
            backEnt.playAnimation(anim.repeat(duration: .infinity), transitionDuration: 1.25, startsPaused: false)
        }
        
        // 赤い対角線を描画するためのUIView
        let lineView = DiagonalLineView()
        lineView.backgroundColor = .clear
        lineView.translatesAutoresizingMaskIntoConstraints = false
        arview.addSubview(lineView)
        
        // lineViewの制約を追加
        NSLayoutConstraint.activate([
            lineView.leadingAnchor.constraint(equalTo: arview.leadingAnchor),
            lineView.trailingAnchor.constraint(equalTo: arview.trailingAnchor),
            lineView.topAnchor.constraint(equalTo: arview.topAnchor),
            lineView.bottomAnchor.constraint(equalTo: arview.bottomAnchor)
        ])
    }
}

// 赤い対角線を描画するためのUIView
class DiagonalLineView: UIView {
    override func draw(_ rect: CGRect) {
        guard let context = UIGraphicsGetCurrentContext() else { return }
        context.setStrokeColor(UIColor.red.cgColor)
        context.setLineWidth(2.0)
        
        // 左上から右下への対角線
        context.move(to: CGPoint(x: 0, y: 0))
        context.addLine(to: CGPoint(x: rect.width, y: rect.height))
        context.strokePath()
        
        // 右上から左下への対角線
        context.move(to: CGPoint(x: rect.width, y: 0))
        context.addLine(to: CGPoint(x: 0, y: rect.height))
        context.strokePath()
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        setNeedsDisplay() // サイズ変更時に再描画を要求
    }
}

PlaygroundPage.current.liveView = ARViewController()
PlaygroundPage.current.needsIndefiniteExecution = true

//#-end-editable-code
//#-hidden-code
//#-end-hidden-code
