//
//  See LICENSE folder for this template’s licensing information.
//
//  Abstract:
//  Instantiates a live view and passes it to the PlaygroundSupport framework.
//

import SwiftUI
import CoreMotion
import PlaygroundSupport
import AudioToolbox

//#-editable-code
let degreeValue = 0.1
//#-end-editable-code

class MotionManager: ObservableObject {
    private var motionManager = CMMotionManager()
    
    @Published var pitch: Double = 0.0
    @Published var roll: Double = 0.0
    @Published var isTilted: Bool = false
    
    init() {
        if motionManager.isDeviceMotionAvailable {
            motionManager.deviceMotionUpdateInterval = 0.1
            motionManager.startDeviceMotionUpdates(to: .main) { (motion, error) in
                if let motion = motion {
                    let attitude = motion.attitude
                    
                    // デバイスの向きを取得
                    let orientation = UIDevice.current.orientation
                    
                    // iPadがデフォルトで横向きと仮定
                    switch orientation {
                    case .landscapeLeft:
                        // 左横向き
                        self.pitch = attitude.roll
                        self.roll = -attitude.pitch
                    case .landscapeRight:
                        // 右横向き
                        self.pitch = -attitude.roll
                        self.roll = attitude.pitch
                    case .portrait:
                        // 通常の縦向き
                        self.pitch = attitude.pitch
                        self.roll = attitude.roll
                    case .portraitUpsideDown:
                        // 上下逆さまの縦向き
                        self.pitch = -attitude.pitch
                        self.roll = -attitude.roll
                    default:
                        // デフォルトでは横向きの設定にする
                        self.pitch = attitude.roll
                        self.roll = -attitude.pitch
                    }
                    
                    // 角度が3度（約0.0524ラジアン）以上かチェック
                    if abs(self.pitch) > degreeValue || abs(self.roll) > degreeValue {
                        self.playSound()
                        self.isTilted = true
                    } else {
                        self.isTilted = false
                    }
                    
                }
            }
        }
    }
    
    // システムのブザー音を再生する関数
    private func playSound() {
        AudioServicesPlaySystemSound(SystemSoundID(1005)) // システムのビープ音
    }
}
