//
//  DatabaseManager.swift
//  PlaygroundBook
//
//  Created by 日々野清高 on 2025/05/22.
//

import Foundation
import SQLite3

public class DatabaseManager {
    let dbPath: String
    var db: OpaquePointer?

    public init(dbPath: String) {
        self.dbPath = dbPath
        if sqlite3_open(dbPath, &db) != SQLITE_OK {
            print("データベースを開けませんでした")
        }
    }

    deinit {
        sqlite3_close(db)
    }

    public func execute(sql: String) -> String {
        let trimmed = sql.trimmingCharacters(in: .whitespacesAndNewlines).lowercased()
        if trimmed.hasPrefix("select") {
            return executeSelect(sql: sql)
        } else {
            return executeNonQuery(sql: sql)
        }
    }

    private func executeSelect(sql: String) -> String {
        var stmt: OpaquePointer?
        var result = ""

        if sqlite3_prepare_v2(db, sql, -1, &stmt, nil) == SQLITE_OK {
            let columnCount = sqlite3_column_count(stmt)
            // カラム名の取得
            for i in 0..<columnCount {
                let name = String(cString: sqlite3_column_name(stmt, i))
                result += "\(name)\t"
            }
            result += "\n"

            // 行データの取得
            while sqlite3_step(stmt) == SQLITE_ROW {
                for i in 0..<columnCount {
                    if let cStr = sqlite3_column_text(stmt, i) {
                        let val = String(cString: cStr)
                        result += "\(val)\t"
                    } else {
                        result += "NULL\t"
                    }
                }
                result += "\n"
            }
            sqlite3_finalize(stmt)
            return result
        } else {
            let errMsg = String(cString: sqlite3_errmsg(db))
            return "SELECT 実行失敗: \(errMsg)"
        }
    }

    private func executeNonQuery(sql: String) -> String {
        var errMsg: UnsafeMutablePointer<Int8>?
        if sqlite3_exec(db, sql, nil, nil, &errMsg) == SQLITE_OK {
            return "実行成功"
        } else {
            let message = String(cString: errMsg!)
            sqlite3_free(errMsg)
            return "実行失敗: \(message)"
        }
    }
    
    public func getAllTableNames() -> [String] {
        var tables = [String]()
        let sql = "SELECT name FROM sqlite_master WHERE type='table' AND name NOT LIKE 'sqlite_%';"
        var stmt: OpaquePointer?
        if sqlite3_prepare_v2(db, sql, -1, &stmt, nil) == SQLITE_OK {
            while sqlite3_step(stmt) == SQLITE_ROW {
                if let cStr = sqlite3_column_text(stmt, 0) {
                    let name = String(cString: cStr)
                    tables.append(name)
                }
            }
        }
        sqlite3_finalize(stmt)
        return tables
    }

    public func getColumnNames(table: String) -> [String] {
        var columns = [String]()
        let sql = "PRAGMA table_info(\(table));"
        var stmt: OpaquePointer?
        if sqlite3_prepare_v2(db, sql, -1, &stmt, nil) == SQLITE_OK {
            while sqlite3_step(stmt) == SQLITE_ROW {
                if let cStr = sqlite3_column_text(stmt, 1) {  // 1番目がカラム名
                    let columnName = String(cString: cStr)
                    columns.append(columnName)
                }
            }
        }
        sqlite3_finalize(stmt)
        return columns
    }

    public func getTableData(table: String) -> [[String]] {
        var rows = [[String]]()
        let sql = "SELECT * FROM \(table);"
        var stmt: OpaquePointer?
        if sqlite3_prepare_v2(db, sql, -1, &stmt, nil) == SQLITE_OK {
            let columnCount = sqlite3_column_count(stmt)
            while sqlite3_step(stmt) == SQLITE_ROW {
                var row = [String]()
                for i in 0..<columnCount {
                    if let cStr = sqlite3_column_text(stmt, i) {
                        let val = String(cString: cStr)
                        row.append(val)
                    } else {
                        row.append("NULL")
                    }
                }
                rows.append(row)
            }
        }
        sqlite3_finalize(stmt)
        return rows
    }
    
    public func getColumnTypes(table: String) -> [String] {
        var types = [String]()
        let sql = "PRAGMA table_info(\(table));"
        var stmt: OpaquePointer?
        if sqlite3_prepare_v2(db, sql, -1, &stmt, nil) == SQLITE_OK {
            while sqlite3_step(stmt) == SQLITE_ROW {
                if let cStr = sqlite3_column_text(stmt, 2) {  // 2番目が型名
                    let typeName = String(cString: cStr)
                    types.append(typeName)
                } else {
                    types.append("")
                }
            }
        }
        sqlite3_finalize(stmt)
        return types
    }
}
