//
//  SQLViewController.swift
//  PlaygroundBook
//
//  Created by 日々野清高 on 2025/05/22.
//
import UIKit
import PlaygroundSupport

import UIKit
import PlaygroundSupport

public class SQLViewController: UIViewController, PlaygroundLiveViewMessageHandler, UITableViewDataSource, UITableViewDelegate {

    var dbManager: DatabaseManager?

    let dbNameLabel = UILabel()
    let sqlInputView = UITextView()
    let runButton = UIButton(type: .system)
    let resultView = UITextView()
    let contentScrollView = UIScrollView()
    
    let tablesTableView = UITableView()
    let contentTableView = UITableView()
    let sqlLabel = UILabel()
    let resultLabel = UILabel()
    let tableListLabel = UILabel()
    let contentLabel = UILabel()
    
    var tableNames: [String] = []
    var selectedTableColumns: [String] = []
    var selectedTableRows: [[String]] = []
    var selectedTableColumnTypes: [String] = []

    public func receive(_ message: PlaygroundValue) {
        guard case let .dictionary(dict) = message,
              case let .string(dbPath) = dict["dbPath"] else {
            print("受信メッセージの形式が不正です")
            return
        }
        dbManager = DatabaseManager(dbPath: dbPath)
        DispatchQueue.main.async {
            self.dbNameLabel.text = "接続中: \(URL(fileURLWithPath: dbPath).lastPathComponent)"
            self.loadTableNames()
        }
    }

    override public func viewDidLoad() {
        super.viewDidLoad()
        setupUI()
    }

    func setupUI() {
        // view.backgroundColor = .white

        dbNameLabel.text = "データベース未接続"
        dbNameLabel.font = UIFont.boldSystemFont(ofSize: 20)
        dbNameLabel.translatesAutoresizingMaskIntoConstraints = false

        sqlInputView.font = UIFont.systemFont(ofSize: 16)
        sqlInputView.layer.borderWidth = 1
        sqlInputView.layer.borderColor = UIColor.gray.cgColor
        sqlInputView.layer.cornerRadius = 8
        sqlInputView.translatesAutoresizingMaskIntoConstraints = false

        runButton.setTitle("実行", for: .normal)
        runButton.titleLabel?.font = UIFont.boldSystemFont(ofSize: 18)
        runButton.addTarget(self, action: #selector(runSQL), for: .touchUpInside)
        runButton.translatesAutoresizingMaskIntoConstraints = false

        resultView.font = UIFont.systemFont(ofSize: 14)
        resultView.isEditable = false
        resultView.layer.borderWidth = 1
        resultView.layer.borderColor = UIColor.gray.cgColor
        resultView.layer.cornerRadius = 8
        resultView.translatesAutoresizingMaskIntoConstraints = false

        setupTableView(tablesTableView, identifier: "TableNameCell")
        setupTableView(contentTableView, identifier: "ContentCell")

        view.addSubview(dbNameLabel)
        view.addSubview(sqlInputView)
        view.addSubview(runButton)
        view.addSubview(resultView)
        view.addSubview(tablesTableView)

        // contentTableView を scrollView に入れてから scrollView を view に追加
        contentScrollView.layer.borderWidth = 1
        contentScrollView.layer.borderColor = UIColor.gray.cgColor
        contentScrollView.layer.cornerRadius = 0
        contentScrollView.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(contentScrollView)
        contentScrollView.addSubview(contentTableView)
        contentTableView.translatesAutoresizingMaskIntoConstraints = false

        // 実行ボタンを装飾
        runButton.layer.borderWidth = 1
        runButton.layer.borderColor = UIColor.systemBlue.cgColor
        runButton.layer.cornerRadius = 6
        if #available(iOS 15.0, *) {
            var config = UIButton.Configuration.filled()
            config.title = "実行"
            config.baseForegroundColor = .white
            config.baseBackgroundColor = .systemBlue
            config.contentInsets = NSDirectionalEdgeInsets(top: 6, leading: 12, bottom: 6, trailing: 12)
            runButton.configuration = config
        } else {
            // iOS 13–14 fallback
            runButton.setTitle("実行", for: .normal)
            runButton.setTitleColor(.systemBlue, for: .normal)
            runButton.layer.borderWidth = 1
            runButton.layer.borderColor = UIColor.systemBlue.cgColor
            runButton.layer.cornerRadius = 6
            runButton.contentEdgeInsets = UIEdgeInsets(top: 6, left: 12, bottom: 6, right: 12)
        }

        // ラベルをビューに追加
        sqlLabel.text = "SQL文"
        sqlLabel.font = .systemFont(ofSize: 10, weight: .bold)
        sqlLabel.translatesAutoresizingMaskIntoConstraints = false
        resultLabel.text = "結果表示"
        resultLabel.font = .systemFont(ofSize: 10, weight: .bold)
        resultLabel.translatesAutoresizingMaskIntoConstraints = false
        tableListLabel.text = "テーブル一覧"
        tableListLabel.font = .systemFont(ofSize: 10, weight: .bold)
        tableListLabel.translatesAutoresizingMaskIntoConstraints = false
        contentLabel.text = "テーブルの内容"
        contentLabel.font = .systemFont(ofSize: 10, weight: .bold)
        contentLabel.translatesAutoresizingMaskIntoConstraints = false
        
        view.addSubview(sqlLabel)
        view.addSubview(resultLabel)
        view.addSubview(tableListLabel)
        view.addSubview(contentLabel)
        
        setupConstraints()
    }
    
    func setupTableView(_ tableView: UITableView, identifier: String) {
        tableView.translatesAutoresizingMaskIntoConstraints = false
        tableView.dataSource = self
        tableView.delegate = self
        if identifier == "ContentCell" {
            tableView.register(TableRowCell.self, forCellReuseIdentifier: identifier)
        } else {
            tableView.register(UITableViewCell.self, forCellReuseIdentifier: identifier)
            tableView.layer.borderWidth = 1
            tableView.layer.borderColor = UIColor.gray.cgColor
            tableView.layer.cornerRadius = 0
        }
    }


    func setupConstraints() {
        NSLayoutConstraint.activate([
            dbNameLabel.topAnchor.constraint(equalTo: view.safeAreaLayoutGuide.topAnchor, constant: 10),
            dbNameLabel.centerXAnchor.constraint(equalTo: view.centerXAnchor),

            sqlLabel.topAnchor.constraint(equalTo: dbNameLabel.bottomAnchor, constant: 10),
            sqlLabel.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            
            sqlInputView.topAnchor.constraint(equalTo: sqlLabel.bottomAnchor, constant: 5),
            sqlInputView.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            sqlInputView.trailingAnchor.constraint(equalTo: view.trailingAnchor, constant: -10),
            sqlInputView.heightAnchor.constraint(equalToConstant: 120),

            runButton.topAnchor.constraint(equalTo: sqlInputView.bottomAnchor, constant: 10),
            runButton.centerXAnchor.constraint(equalTo: view.centerXAnchor),

            resultLabel.topAnchor.constraint(equalTo: runButton.bottomAnchor, constant: 10),
            resultLabel.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            
            resultView.topAnchor.constraint(equalTo: resultLabel.bottomAnchor, constant: 5),
            resultView.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            resultView.trailingAnchor.constraint(equalTo: view.trailingAnchor, constant: -10),
            resultView.heightAnchor.constraint(equalToConstant: 80),

            tableListLabel.topAnchor.constraint(equalTo: resultView.bottomAnchor, constant: 10),
            tableListLabel.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            
            tablesTableView.topAnchor.constraint(equalTo: tableListLabel.bottomAnchor, constant: 5),
            tablesTableView.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 10),
            tablesTableView.widthAnchor.constraint(equalToConstant: 160),
            tablesTableView.bottomAnchor.constraint(equalTo: view.safeAreaLayoutGuide.bottomAnchor, constant: -10),

            contentLabel.topAnchor.constraint(equalTo: resultView.bottomAnchor, constant: 10),
            contentLabel.leadingAnchor.constraint(equalTo: tablesTableView.trailingAnchor, constant: 10),

            // contentScrollView 制約
            contentScrollView.topAnchor.constraint(equalTo: contentLabel.bottomAnchor, constant: 5),
            contentScrollView.leadingAnchor.constraint(equalTo: tablesTableView.trailingAnchor, constant: 10),
            contentScrollView.trailingAnchor.constraint(equalTo: view.trailingAnchor, constant: -10),
            contentScrollView.bottomAnchor.constraint(equalTo: view.safeAreaLayoutGuide.bottomAnchor, constant: -10),

            // contentTableView 制約 (scrollView の中)
            contentTableView.topAnchor.constraint(equalTo: contentScrollView.topAnchor),
            contentTableView.leadingAnchor.constraint(equalTo: contentScrollView.leadingAnchor),
            contentTableView.trailingAnchor.constraint(equalTo: contentScrollView.trailingAnchor),
            contentTableView.bottomAnchor.constraint(equalTo: contentScrollView.bottomAnchor),
            contentTableView.widthAnchor.constraint(equalToConstant: 600),

            // 追加：高さ固定
            contentTableView.heightAnchor.constraint(equalTo: contentScrollView.heightAnchor)

        ])
    }



    @objc func runSQL() {
        let sql = sqlInputView.text ?? ""
        let result = dbManager?.execute(sql: sql) ?? "データベース未接続"
        resultView.text += "\n\(result)"
        resultView.scrollRangeToVisible(NSRange(location: resultView.text.count, length: 0))
        loadTableNames()
    }

    func loadTableNames() {
        guard let db = dbManager else { return }
        tableNames = db.getAllTableNames()
        tablesTableView.reloadData()
    }

    func loadTableContent(tableName: String) {
        guard let db = dbManager else { return }
        selectedTableColumns = db.getColumnNames(table: tableName)
        selectedTableColumnTypes = db.getColumnTypes(table: tableName)  // ここで型を取得
        selectedTableRows = db.getTableData(table: tableName)
        contentTableView.reloadData()
    }

    // MARK: - UITableViewDataSource

    public func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }

    public func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        if tableView == tablesTableView {
            return tableNames.count
        } else if tableView == contentTableView {
            return selectedTableRows.count + 1
        }
        return 0
    }

    public func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        if tableView == tablesTableView {
            let cell = tableView.dequeueReusableCell(withIdentifier: "TableNameCell", for: indexPath)
            cell.textLabel?.text = tableNames[indexPath.row]
            return cell
        } else if tableView == contentTableView {
            guard let cell = tableView.dequeueReusableCell(withIdentifier: "ContentCell", for: indexPath) as? TableRowCell else {
                return UITableViewCell()
            }

            let columnCount = max(1, selectedTableColumns.count)
            let columnWidth: CGFloat = tableView.bounds.width / CGFloat(columnCount)

            if indexPath.row == 0 {
                let headerWithTypes = zip(selectedTableColumns, selectedTableColumnTypes).map { "\($0)\n[\($1)]" }
                cell.configure(columns: headerWithTypes, columnWidth: columnWidth, isHeader: true)
            } else {
                let row = selectedTableRows[indexPath.row - 1]
                let paddedRow = row + Array(repeating: "", count: max(0, selectedTableColumns.count - row.count))
                cell.configure(columns: paddedRow, columnWidth: columnWidth, isHeader: false)
            }

            return cell
        }
        return UITableViewCell()
    }


    // MARK: - UITableViewDelegate

    public func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        if tableView == tablesTableView {
            let selectedTable = tableNames[indexPath.row]
            loadTableContent(tableName: selectedTable)
        }
    }
}

// MARK: - カスタムセル（1行に複数のカラムをラベル表示）
class TableRowCell: UITableViewCell {
    var columnLabels: [UILabel] = []

    func configure(columns: [String], columnWidth: CGFloat, isHeader: Bool = false) {
        columnLabels.forEach { $0.removeFromSuperview() }
        columnLabels = []

        for (index, text) in columns.enumerated() {
            let label = UILabel()
            label.text = text

            if isHeader {
                // ヘッダは太字＆中央寄せ
                label.font = UIFont.boldSystemFont(ofSize: 14)
                label.textAlignment = .center
            } else {
                label.font = UIFont.monospacedSystemFont(ofSize: 14, weight: .regular)
                // 数値かどうか判定（簡易）
                if let _ = Double(text) {
                    label.textAlignment = .right
                } else {
                    label.textAlignment = .left
                }
            }

            label.layer.borderWidth = 0.5
            label.layer.borderColor = UIColor.lightGray.cgColor
            label.numberOfLines = 0
            label.lineBreakMode = .byWordWrapping
            label.frame = CGRect(x: CGFloat(index) * columnWidth, y: 0, width: columnWidth, height: self.contentView.bounds.height)
            label.autoresizingMask = [.flexibleHeight]
            self.contentView.addSubview(label)
            columnLabels.append(label)
        }
    }

    override func prepareForReuse() {
        super.prepareForReuse()
        columnLabels.forEach { $0.removeFromSuperview() }
        columnLabels = []
    }
}
