//
//  See LICENSE folder for this template’s licensing information.
//
//  Abstract:
//  Contains classes/structs/enums/functions which are part of a module that is
//  automatically imported into user-editable code.
//

import BookCore

// Implement any classes/structs/enums/functions in the BookAPI module which you
// want to be automatically imported and visible for users on playground pages
// and in user modules.
//
// This is controlled via the book-level `UserAutoImportedAuxiliaryModules`
// Manifest.plist key.


import PlaygroundSupport
import UIKit

public var unitSize: Double = 50.0  // 1マス=50ピクセル（初期値）
private var commandQueue: [() -> Void] = []
private var isBusy = false

private let page = PlaygroundPage.current

public func setupPlotterAPI() {
    page.needsIndefiniteExecution = true
    if let proxy = page.liveView as? PlaygroundRemoteLiveViewProxy {
        proxy.delegate = PlotterDelegate.shared
    }
}

private func enqueue(_ command: @escaping () -> Void) {
    commandQueue.append(command)
    if !isBusy {
        runNext()
    }
}

private func runNext() {
    guard !commandQueue.isEmpty else {
        return
    }
    isBusy = true
    let next = commandQueue.removeFirst()
    next()
}

private class PlotterDelegate: NSObject, PlaygroundRemoteLiveViewProxyDelegate {
    static let shared = PlotterDelegate()
    func remoteLiveViewProxy(_ proxy: PlaygroundRemoteLiveViewProxy, received message: PlaygroundValue) {
        if case .string(let str) = message, str == "done" {
            isBusy = false
            runNext()
        }
    }

    func remoteLiveViewProxyConnectionClosed(_ proxy: PlaygroundRemoteLiveViewProxy) {
        // 接続が閉じられたときの処理（必要に応じて追加）
    }
}

private func send(_ value: PlaygroundValue) {
    if let proxy = page.liveView as? PlaygroundRemoteLiveViewProxy {
        proxy.send(value)
    }
}

// MARK: - 公開API関数

public func setUnit(_ value: Any) {
    let inputValue = (value as? Double) ?? Double("\(value)") ?? 0.0
    let clampedValue = min(max(inputValue, 1.0), 100.0)
    enqueue {
        unitSize = clampedValue
    }
}

public func setAngle(_ value: Any) {
    let angleValue = (value as? Double) ?? Double("\(value)") ?? 0.0
    enqueue {
        send(.dictionary([
            "command": .string("angle"),
            "value": .floatingPoint(angleValue)
        ]))
    }
}

public func turn(_ angle: Any) {
    let angleValue = (angle as? Double) ?? Double("\(angle)") ?? 0.0
    enqueue {
        send(.dictionary([
            "command": .string("turn"),
            "angle": .floatingPoint(angleValue)
        ]))
    }
}

public func move(_ distance: Any) {
    let distanceValue = (distance as? Double) ?? Double("\(distance)") ?? 0.0
    guard distanceValue >= 0 else {
        debugPrint("警告: move() に負の値は使えません")
        return
    }

    let pixelDistance = distanceValue * unitSize
    enqueue {
        send(.dictionary([
            "command": .string("move"),
            "distance": .floatingPoint(pixelDistance)
        ]))
    }
}

public func moveTo(_ x: Any, _ y: Any) {
    let xVal = (x as? Double) ?? Double("\(x)") ?? 0.0
    let yVal = (y as? Double) ?? Double("\(y)") ?? 0.0
    enqueue {
        send(.dictionary([
            "command": .string("moveTo"),
            "x": .floatingPoint(xVal * unitSize),
            "y": .floatingPoint(yVal * unitSize)
        ]))
    }
}

public func moveBy(_ x: Any, _ y: Any) {
    let xVal = (x as? Double) ?? Double("\(x)") ?? 0.0
    let yVal = (y as? Double) ?? Double("\(y)") ?? 0.0
    enqueue {
        send(.dictionary([
            "command": .string("moveBy"),
            "x": .floatingPoint(xVal * unitSize),
            "y": .floatingPoint(yVal * unitSize)
        ]))
    }
}

public func circle(_ radius: Any) {
    let rVal = (radius as? Double) ?? Double("\(radius)") ?? 0.0
    enqueue {
        send(.dictionary([
            "command": .string("drawCircle"),
            "radius": .floatingPoint(rVal * unitSize)
        ]))
    }
}

public func speed(_ value: Any) {
    let level = (value as? Int) ?? Int("\(value)") ?? 1
    let clampedLevel = max(1, min(level, 10))
    let speedValue: Double

    switch clampedLevel {
    case 1: speedValue = 100
    case 2: speedValue = 250
    case 3: speedValue = 500
    case 4: speedValue = 1000
    case 5: speedValue = 2000
    case 6: speedValue = 3500
    case 7: speedValue = 5000
    case 8: speedValue = 7000
    case 9: speedValue = 8500
    case 10: speedValue = 9999
    default: speedValue = 100
    }

    enqueue {
        send(.dictionary([
            "command": .string("speed"),
            "value": .floatingPoint(speedValue)
        ]))
    }
}

public func color(_ name: String) {
    enqueue {
        send(.dictionary([
            "command": .string("color"),
            "value": .string(name)
        ]))
    }
}

public func width(_ value: Any) {
    let wVal = (value as? Double) ?? Double("\(value)") ?? 0.0
    let clamped = max(1.0, min(wVal, 10.0))
    enqueue {
        send(.dictionary([
            "command": .string("width"),
            "value": .floatingPoint(clamped)
        ]))
    }
}

public func on() {
    enqueue {
        send(.dictionary(["command": .string("on")]))
    }
}

public func off() {
    enqueue {
        send(.dictionary(["command": .string("off")]))
    }
}

public func clear() {
    enqueue {
        send(.dictionary(["command": .string("clear")]))
    }
}

public func finish() {
    enqueue {
        page.finishExecution()
    }
}

// ストロークデータを格納する配列
public var strokeData: [(startX: Int, startY: Int, endX: Int, endY: Int)] = []

// ファイル読み込み時にクリアを忘れずに
public func loadStrokeData(from filename: String) {
    strokeData.removeAll()
    if let path = Bundle.main.path(forResource: filename, ofType: "txt") {
        do {
            let fileContent = try String(contentsOfFile: path)
            let lines = fileContent.split(separator: "\n")

            for line in lines {
                let components = line.split(separator: ",").map { String($0).trimmingCharacters(in: .whitespaces) }
                if components.count == 4,
                   let startX = Int(components[0]),
                   let startY = Int(components[1]),
                   let endX = Int(components[2]),
                   let endY = Int(components[3]) {
                    strokeData.append((startX, startY, endX, endY))
                }
            }
        } catch {
            print("Error reading file: \(error)")
        }
    }
}

public func executeStrokeData(_ x: Any, _ y: Any) {
    // 引数をDouble型またはInt型に変換
    let xValue = (x as? Double) ?? Double("\(x)") ?? Double((x as? Int) ?? 0)
    let yValue = (y as? Double) ?? Double("\(y)") ?? Double((y as? Int) ?? 0)
    let pixelX = xValue * unitSize
    let pixelY = yValue * unitSize
    for stroke in strokeData {
        if stroke.startY % 5 == 0 && stroke.endY % 5 == 0 {
            moveTo(Double(stroke.startX) / unitSize + pixelX / unitSize, -Double(stroke.startY) / unitSize + pixelY / unitSize)
            on()
            moveBy(Double(stroke.endX) / unitSize - Double(stroke.startX) / unitSize,
                   -(Double(stroke.endY) / unitSize - Double(stroke.startY) / unitSize))
            off()
        }
    }
}
